/**
 * Gmail Page Detection Module
 * Detects Gmail pages and identifies page type (single email vs inbox/list)
 */

class GmailDetector {
    constructor() {
        this.isGmailPage = false;
        this.pageType = null; // 'email_view', 'inbox_list', or 'category_view'
        this.emailId = null;
        this.threadId = null;
        this.category = null; // Gmail category (e.g., 'purchases', 'social', 'promotions')
    }

    /**
     * Detect if current page is Gmail
     */
    isGmail() {
        const hostname = window.location.hostname.toLowerCase();
        return hostname === 'mail.google.com' || hostname.endsWith('.mail.google.com');
    }

    /**
     * Detect Gmail page type and extract context
     * Returns: { isGmail, pageType, emailId, threadId }
     */
    detectPage() {
        this.isGmailPage = this.isGmail();

        if (!this.isGmailPage) {
            void 0;
            return {
                isGmail: false,
                pageType: null,
                emailId: null,
                threadId: null
            };
        }

        void 0;

        // Parse URL hash to determine page type and extract IDs
        const hash = window.location.hash;
        void 0;

        // Category view: #category/purchases, #category/social, etc.
        const categoryViewMatch = hash.match(/#category\/([a-z]+)(?:\/([a-f0-9]+))?$/);

        if (categoryViewMatch) {
            const category = categoryViewMatch[1];
            const messageId = categoryViewMatch[2];

            if (messageId) {
                // Category email view: #category/purchases/message_id
                this.pageType = 'email_view';
                this.category = category;
                this.emailId = messageId;
                this.threadId = messageId;

                void 0;

                return {
                    isGmail: true,
                    pageType: 'email_view',
                    category: this.category,
                    emailId: this.emailId,
                    threadId: this.threadId
                };
            } else {
                // Category list view: #category/purchases
                this.pageType = 'category_view';
                this.category = category;

                void 0;

                return {
                    isGmail: true,
                    pageType: 'category_view',
                    category: this.category,
                    emailId: null,
                    threadId: null
                };
            }
        }

        // Email view: #inbox/message_id or #all/message_id
        // Thread view: #inbox/thread_id (multiple messages)
        const emailViewMatch = hash.match(/#[^/]+\/([a-f0-9]+)$/);

        if (emailViewMatch) {
            // Single email or thread view
            const id = emailViewMatch[1];

            // Try to determine if this is a single email or thread
            // Gmail uses different ID formats, but we'll treat both as potential email views
            this.pageType = 'email_view';
            this.emailId = id;
            this.threadId = id; // May be thread or message ID

            void 0;

            return {
                isGmail: true,
                pageType: 'email_view',
                emailId: this.emailId,
                threadId: this.threadId
            };
        }

        // Inbox/list view: #inbox, #all, #sent, etc.
        const listViewMatch = hash.match(/#([^/]+)$/);

        if (listViewMatch) {
            this.pageType = 'inbox_list';

            void 0;

            return {
                isGmail: true,
                pageType: 'inbox_list',
                emailId: null,
                threadId: null
            };
        }

        // Default: treat as list view if on Gmail but can't parse
        this.pageType = 'inbox_list';
        void 0;

        return {
            isGmail: true,
            pageType: 'inbox_list',
            emailId: null,
            threadId: null
        };
    }

    /**
     * Get current page info
     */
    getPageInfo() {
        if (!this.isGmailPage) {
            this.detectPage();
        }

        return {
            isGmail: this.isGmailPage,
            pageType: this.pageType,
            category: this.category,
            emailId: this.emailId,
            threadId: this.threadId
        };
    }

    /**
     * Check if on single email view
     */
    isEmailView() {
        if (!this.isGmailPage) {
            this.detectPage();
        }
        return this.pageType === 'email_view';
    }

    /**
     * Check if on inbox/list view
     */
    isInboxView() {
        if (!this.isGmailPage) {
            this.detectPage();
        }
        return this.pageType === 'inbox_list';
    }

    /**
     * Check if on category view (e.g., #category/purchases)
     */
    isCategoryView() {
        if (!this.isGmailPage) {
            this.detectPage();
        }
        return this.pageType === 'category_view';
    }

    /**
     * Get current category (if on category view)
     */
    getCurrentCategory() {
        return this.category;
    }

    /**
     * Get email ID from current page (single email view only)
     */
    getCurrentEmailId() {
        if (!this.isEmailView()) {
            return null;
        }
        return this.emailId;
    }

    /**
     * Get thread ID from current page
     */
    getCurrentThreadId() {
        return this.threadId;
    }

    /**
     * Extract message/thread ID from current page URL (works for popout windows and regular view)
     * Returns: message ID string or null
     */
    getCurrentMessageId() {
        const url = window.location.href;
        const hash = window.location.hash;

        void 0;

        // Method 1: Extract from URL hash (regular Gmail view)
        // Format: #inbox/18abc123def or #all/18abc123def
        const hashMatch = hash.match(/#[^/]+\/([a-f0-9]+)$/);
        if (hashMatch && hashMatch[1]) {
            void 0;
            return hashMatch[1];
        }

        // Method 2: Extract from popout URL query parameter
        // Format: mail.google.com/mail/u/0/popout?ver=...&th=%23thread-f%3A1846807978230534613
        const urlParams = new URLSearchParams(window.location.search);
        const threadParam = urlParams.get('th');

        if (threadParam) {
            // Thread param is URL-encoded: %23thread-f%3A1846807978230534613
            // Decode it: #thread-f:1846807978230534613
            const decodedThread = decodeURIComponent(threadParam);

            // Extract numeric ID from formats like:
            // #thread-f:1846807978230534613
            // #msg-f:1846807978230534613
            const threadIdMatch = decodedThread.match(/[:#](?:thread|msg)-[a-z][:]*(\d+)/);
            if (threadIdMatch && threadIdMatch[1]) {
                void 0;
                return threadIdMatch[1];
            }

            // Fallback: try extracting any long numeric ID
            const numericMatch = decodedThread.match(/(\d{15,})/);
            if (numericMatch && numericMatch[1]) {
                void 0;
                return numericMatch[1];
            }
        }

        // Method 3: Check for message_id parameter (some Gmail URLs)
        const msgId = urlParams.get('message_id') || urlParams.get('msg');
        if (msgId) {
            void 0;
            return msgId;
        }

        void 0;
        return null;
    }

    /**
     * Check if current page is a Gmail popout window
     */
    isPopoutWindow() {
        const url = window.location.href;
        const isPopout = url.includes('/popout');
        void 0;
        return isPopout;
    }

    /**
     * Extract visible emails from inbox/list view
     * Returns array of message IDs visible on current page
     */
    getVisibleEmailIds() {
        if (!this.isInboxView()) {
            void 0;
            return [];
        }

        // Gmail's UI is heavily dynamic - this is a best-effort extraction
        // Look for email rows in the inbox
        const emailRows = document.querySelectorAll('[data-message-id], [data-thread-id], tr.zA');

        const emailIds = [];
        emailRows.forEach(row => {
            // Try multiple attributes that Gmail uses
            const messageId = row.getAttribute('data-message-id') ||
                            row.getAttribute('data-thread-id') ||
                            row.getAttribute('id');

            if (messageId && messageId.length > 5) {
                emailIds.push(messageId);
            }
        });

        void 0;
        return emailIds;
    }

    /**
     * Estimate total number of messages in current view
     * This is approximate - Gmail doesn't expose exact counts easily
     */
    estimateMessageCount() {
        // Look for Gmail's message count display
        const countElements = document.querySelectorAll('[aria-label*="messages"], .Dj .T-I-ax7');

        for (const elem of countElements) {
            const text = elem.textContent || elem.getAttribute('aria-label') || '';
            const match = text.match(/(\d+)\s*(messages?|emails?)/i);
            if (match) {
                const count = parseInt(match[1], 10);
                void 0;
                return count;
            }
        }

        // Fallback: count visible rows
        const visibleCount = this.getVisibleEmailIds().length;
        void 0;
        return visibleCount;
    }
}

// Export for use in content script
if (typeof module !== 'undefined' && module.exports) {
    module.exports = GmailDetector;
}
